package Musings::Author;

use Musings::DB;
use Carp qw(croak);
use strict;

=head1 NAME

Musings::Author

=head1 SYNOPSIS

  use Musings::Author;
  my $author = Musings::Author->new;
  $author->first_name('Eric');
  $author->middle_name('Lease');
  $author->last_name('Morgan');
  $author->email('eric_morgan@infomotions.com');
  $author->url('http://infomotions.com');
  $author->commit;
  my $id = $author->author_id;
  $author = Musings::Author->new(id => $id);
  my $first_name  = $author->first_name;
  my $middle_name = $author->middle_name;
  my $last_name   = $author->last_name;
  my $email       = $author->email;
  my $url         = $author->url;
  $author->delete;
  my @authors = Musings::Author->get_authors;


=head1 DESCRIPTION

This package does the necessary I/O against the underlying database's authors table.


=head1 METHODS


=head2 new

Creates a new author object. If passed an id parameter plus a valid author ID, then a specific author object will be created.


=head2 first_name

Get and set the author's first name.


=head2 middle_name

Get and set the author's middle name.


=head2 last_name

Get and set the author's last name.


=head2 email

Get and set the author's email address.


=head2 url

Get and set the author's URL of their home page.


=head2 commit

Save the current author object's attributes to the database. Returns true upon success.


=head2 author_id

Gets the current ID of the current author object. You can not set the author_id value.


=head2 delete

Deletes (drops) the current author from the database. Be careful. Your application is responsible for passing the ACID test.


=head2 get_authors

Returns an array of all the Musings::Author objects in the database. The returned array will be sorted by last_name.


=head1 AUTHORS

Eric Lease Morgan <eric_morgan@infomotions.com>

=cut


sub new {

	# declare local variables
	my ($class, %opts) = @_;
	my $self = {};

	# check for an id
	if ($opts{id}) {
	
		my $dbh = Musings::DB->dbh();
		my $rv = $dbh->selectrow_hashref('SELECT * FROM authors WHERE author_id = ?', undef, $opts{id});
		if (ref($rv) eq "HASH") { 
			$self = $rv;
		} else { 
			return; 
		}
	} elsif ($opts{name}) {
		my $dbh = Musings::DB->dbh();
		my $rv = $dbh->selectrow_hashref('SELECT * FROM authors WHERE last_name = ?', undef, $opts{name});
		if (ref($rv) eq "HASH") { 
			$self = $rv;
		} else { 
			return;
		} 
	}
	
	# return the object
	return bless $self, $class;
	
}


sub author_id {

	my $self = shift;
	return $self->{author_id};

}


sub first_name {

	# declare local variables
	my ($self, $first_name) = @_;
	
	# check for the existance of a note 
	if ($first_name) { $self->{first_name} = $first_name }
	
	# return the name
	return $self->{first_name};
	
}


sub middle_name {

	# declare local variables
	my ($self, $middle_name) = @_;
	
	# check for the existance of a note 
	if ($middle_name) { $self->{middle_name} = $middle_name }
	
	# return the name
	return $self->{middle_name};
	
}


sub last_name {

	# declare local variables
	my ($self, $last_name) = @_;
	
	# check for the existance of a note 
	if ($last_name) { $self->{last_name} = $last_name }
	
	# return the name
	return $self->{last_name};
	
}


sub email {

	# declare local variables
	my ($self, $email) = @_;
	
	# check for the existance of a note 
	if ($email) { $self->{email} = $email }
	
	# return the name
	return $self->{email};
	
}


sub url {

	# declare local variables
	my ($self, $url) = @_;
	
	# check for the existance of a note 
	if ($url) { $self->{url} = $url }
	
	# return the name
	return $self->{url};
	
}


sub commit {

	# get myself, :-)
	my $self = shift;
	
	# get a database handle
	my $dbh = Musings::DB->dbh();	
	
	# see if the object has an id
	if ($self->author_id()) {
	
		# update the record with this id
		my $return = $dbh->do('UPDATE authors SET first_name = ?, middle_name = ?, last_name = ?, email = ?, url = ? WHERE author_id = ?', undef, $self->first_name(), $self->middle_name(), $self->last_name(), $self->email(), $self->url(), $self->author_id());
		if ($return > 1 || ! $return) { croak "Author update in commit() failed. $return records were updated." }
	
	}
	
	else {
	
		# get a new sequence
		my $id = Musings::DB->nextID();
		
		# create a new record
		my $return = $dbh->do('INSERT INTO authors (author_id, first_name, middle_name, last_name, email, url) VALUES (?, ?, ?, ?, ?, ?)', undef, $id, $self->first_name, $self->middle_name, $self->last_name, $self->email, $self->url);
		if ($return > 1 || ! $return) { croak 'Author commit() failed.'; }
		$self->{author_id} = $id;
		
	}
	
	# done
	return 1;
	
}


sub delete {

	my $self = shift;

	if ($self->{author_id}) {

		my $dbh = Musings::DB->dbh();

		# now, delete the primary facet record
		my $rv = $dbh->do('DELETE FROM authors WHERE author_id = ?', undef, $self->{author_id});
		if ($rv != 1) {croak ("Error deleting facet record. Deleted $rv records.");}
		 
		return 1;

	}

	return 0;

}

sub get_authors {

	my $self = shift;
	my %opts = @_;
	my @rv   = ();

	# create and execute a query
	my $dbh = Musings::DB->dbh();

	my $author_ids = $dbh->selectcol_arrayref('SELECT author_id FROM authors ORDER BY last_name');
			
	foreach my $author_id (@$author_ids) { push (@rv, Musings::Author->new(id => $author_id)) }	
	
	return @rv;
}



# return true
1;
