#!/usr/bin/perl

# terms.cgi - create and maintain MyLibrary terms

# Eric Lease Morgan <emorgan@nd.edu>
# January 27, 2006 - moving to the MyLibrary distro


=head1 NAME

terms.cgi - create and manage MyLibrary terms


=head1 DESCRIPTION

Use this script to create and manage MyLibrary terms.

MyLibrary information resources are "classified" using facet/term combinations. Facets might be Subjects, Audiences, Formats, Research Tools, etc. Terms might be Astronomy, Philosophy, Freshman, Faculty, Microfiche, Book, Indexes, Catalogs, etc. Terms are the controlled vocabuary of your system and are integral to your system's organization and browsability.

This script requires no configuration.

=head1 AUTHOR

Eric Lease Morgan


=cut


# require the necessary modules
use lib '../lib';
use CGI qw/:standard/;
use CGI::Carp qw(fatalsToBrowser);
use strict;
use MyLibrary::Core;
require 'etc/subroutines.pl';

use constant RESOURCECGI => './resources.cgi';

# initialize two global variables
my $cgi = CGI->new;
my $html = '';

# branch according to the input
if (! $cgi->param('cmd')) { &home }
elsif ($cgi->param('cmd') eq 'create')  { &create }
elsif ($cgi->param('cmd') eq 'find')    { &find }
elsif ($cgi->param('cmd') eq 'edit')    { &edit }
elsif ($cgi->param('cmd') eq 'delete')  { &delete }
else { $html = $cgi->p("Unknown command. Call Eric") }

# quit
&gracefulExit;


sub header {
	
	# get the value of this script
	my $script = $cgi->script_name;

	# return a standard html page; notice the stylesheet
	return <<EOH;
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
	<head>
		<title>Terms</title>
		<link rel='stylesheet' href='../etc/portal-screen.css' type='text/css' />
	</head>
	<body>
		<div id="navcontainer">
			<ul id="navlist">
				<li><a href="admin.cgi">Home</a></li>
				<li><a href="facets.cgi">Facets</a></li>
				<li><a href="terms.cgi" id="current">Terms</a></li>
				<li><a href="location-types.cgi">Location types</a></li>
				<li><a href="resources.cgi">Resources</a></li>
				<li><a href="patrons.cgi">Patrons</a></li>
			</ul>
		</div>
	
	<div class='menu'>
		<p><b><a href="$script">Manage terms</a></b></p>
		<a href="$script?cmd=create">Create</a><br />
		<a href="$script?cmd=find">Find all</a><br />
		<a href="$script?cmd=edit">Edit</a><br />
		<a href="$script?cmd=delete">Delete</a><br />
		<br />
	</div>
	<div class='content'>
EOH

}


sub home {

	# build a simple home page
	$html  = $cgi->h1('Manage terms');
	$html .= $cgi->p('Use these pages to manage MyLibrary Portal terms.');

}


sub create {

	# initialize the output
	$html = $cgi->h1('Create');
	
	# branch according to the input
	if (! $cgi->param('button')) {
		
		# create a hash of terms; needs to be sorted
		my @facets = MyLibrary::Facet->get_facets(sort => 'name');
		my %facets;
		foreach (@facets) { $facets{$_->facet_id} = $_->facet_name }

		# build the form
		$html .= $cgi->p('Use this form to create a term.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'create');
		$html .= table({-border => 0}, Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item', 'Value']),
							td(['Term', $cgi->textfield(-name => 'term_name')]),
							td(['Facet', $cgi->popup_menu(-name => 'facet_name', -values => [sort values %facets])]),
							td(['Note', $cgi->textarea(-name => 'term_note')]),
							td([''    , $cgi->submit(-name => 'button', -value => 'Create')])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
    else {
    
    	# create an object, fill it, and save
    	my $term = MyLibrary::Term->new;
    	$term->term_name($cgi->param('term_name'));
    	$term->term_note($cgi->param('term_note'));
    	my $facet = MyLibrary::Facet->new(name => $cgi->param('facet_name'));
    	$term->facet_id($facet->facet_id);
		$term->commit;
		
		# get the facet name
		my $facet = MyLibrary::Facet->new(id => $term->facet_id);
		
		# echo the results
		$html .= $cgi->p('Term created:');
		$html .= $cgi->ul(li('ID -', $term->term_id),
		                  li('Facet - ', $facet->facet_name),
		                  li('Term - ', $term->term_name),
		                  li('Note - ', $term->term_note)
		                 );
    
    }
    
}


sub find {

	# initialize the output
	$html  = $cgi->h1('All terms');
	$html .= $cgi->p('This is a list of all the terms:');
	
	# get all the terms, build a list, and display it
	my @terms = MyLibrary::Term->get_terms(sort => 'name');
	my $items;
	foreach (@terms) {
	
		$items .= $cgi->li($cgi->a({-href => $cgi->script_name . '?cmd=edit&button=edit&name=' . $_->term_name}, $_->term_name) . ' - ', $_->term_note, ' (' . $cgi->a({-href => RESOURCECGI . '?cmd=find&field=term&button=Go&query=' . $_->term_name}, scalar($_->related_resources) . ' items') . ')</li>');
	
	}
	$html .= $cgi->ul($items);
	
}


sub edit {

	# initialize the output and a button
	$html = $cgi->h1('Edit terms');
	my $submit .= $cgi->submit(-name => 'button', -value => 'Edit');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# create a hash of terms; needs to be sorted
		my @terms = MyLibrary::Term->get_terms(sort => 'name');
		my %terms;
		foreach (@terms) { $terms{$_->term_id} = $_->term_name }
	
		# display a form
		$html .= $cgi->p('Select a term to edit');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->popup_menu(-name => 'name', -values => [sort values %terms]);
		$html .= $submit;			 
		$html .= $cgi->endform;

	}
	
	elsif (! $cgi->param('term_name') || ! $cgi->param('term_note')) {
	
    	# create an object based on the input, and create form parts
    	my $term = MyLibrary::Term->new(name => $cgi->param('name'));
    	
		# create a hash of terms; needs to be sorted
		my @facets = MyLibrary::Facet->get_facets(sort => 'name');
		my %facets;
		foreach (@facets) { $facets{$_->facet_id} = $_->facet_name }

		# display an edit form
		$html .= $cgi->p('Use this form to edit a term.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->hidden(-name => 'id', -value => $term->term_id);
		$html .= table({-border => 0},
					Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item',  'Value']),
							td(['ID'  ,  $term->term_id]),
							td(['Term',  $cgi->textfield(-name => 'term_name', -value => $term->term_name)]),
							td(['Facet', $cgi->popup_menu(-name => 'facet_name', -values => [sort values %facets], -default => $term->facet_id)]),
							td(['Note',  $cgi->textarea(-name => 'term_note', -value => $term->term_note)]),
							td([''    ,  $submit])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
	else {

		# create and object, fill it, and save
		my $term = MyLibrary::Term->new(id => $cgi->param('id'));
		$term->term_name($cgi->param('term_name'));
		$term->term_note($cgi->param('term_note'));
    	my $facet = MyLibrary::Facet->new(name => $cgi->param('facet_name'));
    	$term->facet_id($facet->facet_id);
		$term->commit;
		
		# get the facet name
		my $facet = MyLibrary::Facet->new(id => $term->facet_id);
		
		# echo the result
		$html .= $cgi->p('Term updated.');
		$html .= $cgi->p('ID:', $term->term_id, '<br />Facet:', $facet->facet_name,'<br />Term:', $term->term_name, '<br />Note:' , $term->term_note);
    
    }

}


sub delete {

	# initialize the output
	$html = $cgi->h1('Delete terms');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# build of hash of all terms; needs to be sorted
		my @terms = MyLibrary::Term->get_terms(sort => 'name');
		my %terms;
		foreach (@terms) { $terms{$_->term_id} = $_->term_name }

		# display a pop-up list of terms
		$html .= $cgi->p('Select a term to delete:');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->popup_menu(-name => 'name', -values => [sort values %terms]);
		$html .= $cgi->submit(-name => 'button', -value => 'delete', -label => 'Delete');		 
		$html .= $cgi->endform;

	}
	
	elsif ($cgi->param('button') eq 'Delete') {
    
    	# create an object based on input
    	my $term = MyLibrary::Term->new(name => $cgi->param('name'));
    	
		# display a confirmation
		$html .= $cgi->p('Are you sure you want to delete the term named <b>' . $term->term_name . '</b>?');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->hidden(-name => 'id', -value => $term->term_id);
		$html .= $cgi->submit(-name => 'button', -value => 'yes', -label => 'Yes');
		$html .= ' ';
		$html .= $cgi->submit(-name => 'button', -value => 'no', -label => 'No');		 
		$html .= $cgi->endform;
    
    }
    
    elsif ($cgi->param('button') eq 'Yes') {
    
    	# delete the term; do the work and echo the result
    	my $term = MyLibrary::Term->new(id => $cgi->param('id'));
    	$term->delete;
    	$html .= $cgi->p('The term ' . $term->term_name . ' has been deleted.');
    
    }
    
    elsif ($cgi->param('button') eq 'No') {
    
    	# abort
    	my $term = MyLibrary::Term->new(id => $cgi->param('id'));
    	$html .= $cgi->p('The term ' . $term->term_name . ' has <b>not</b> been deleted.');

    }
    
}


sub gracefulExit {

	# output the 'magic line', a standard header, the content, and a standard footer, then done
	print $cgi->header;
	print &header;
	print $html;
	print &footer;
	exit;

}



