#!/usr/bin/perl

# facets.cgi - create and maintain MyLibrary facets

# Eric Lease Morgan <emorgan@nd.edu>
# January 27, 2006 - moving to the MyLibrary distro


=head1 NAME

facets.cgi - create and manage MyLibrary facets


=head1 DESCRIPTION

Use this program to create and manage MyLibrary facets.

MyLibrary facets are very broad classifications. An easy example to get your mind around might be Subjects. Other facets might include Audiences, Formats, or Research Tools. Later, using terms.cgi, you will provide sub-classifications (called terms) for these facets. Examples might include Astronomy, Philosophy, Freshman, Seniors, Books, Journals, Dictionaries, Encyclopedias, etc.

This program requires no configuration.


=head1 AUTHOR

Eric Lease Morgan

=cut


# require the necessary modules
use lib '../lib';
use CGI qw/:standard/;
use CGI::Carp qw(fatalsToBrowser);
use strict;
use MyLibrary::Core;
require 'etc/subroutines.pl';

use constant TERMCGI     => './terms.cgi';
use constant RESOURCECGI => './resources.cgi';


# initialize two global variables
my $cgi = CGI->new;
my $html = '';

# branch according to the input
if (! $cgi->param('cmd')) { &home }
elsif ($cgi->param('cmd') eq 'create')  { &create }
elsif ($cgi->param('cmd') eq 'find')    { &find }
elsif ($cgi->param('cmd') eq 'edit')    { &edit }
elsif ($cgi->param('cmd') eq 'delete')  { &delete }
else { $html = $cgi->p("Unknown command. Call Eric.") }

# quit
&gracefulExit;


#############
# subroutines

sub header {
	
	# get the value of this script
	my $script = $cgi->script_name;

	# return a standard html page; notice the stylesheet
	return <<EOH;
<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
<html xmlns="http://www.w3.org/1999/xhtml">
	<head>
		<title>Facets</title>
		<link rel='stylesheet' href='../etc/portal-screen.css' type='text/css' />
	</head>
	<body>
		<div id="navcontainer">
			<ul id="navlist">
				<li><a href="admin.cgi">Home</a></li>
				<li><a href="facets.cgi" id="current">Facets</a></li>
				<li><a href="terms.cgi">Terms</a></li>
				<li><a href="location-types.cgi">Location types</a></li>
				<li><a href="resources.cgi">Resources</a></li>
				<li><a href="patrons.cgi">Patrons</a></li>
			</ul>
		</div>
	
	<div class='menu'>
		<p><b><a href="$script">Manage facets</a></b></p>
		<a href="$script?cmd=create">Create</a><br />
		<a href="$script?cmd=find">Find all</a><br />
		<a href="$script?cmd=edit">Edit</a><br />
		<a href="$script?cmd=delete">Delete</a><br />
		<br />
	</div>
	<div class='content'>
EOH

}


sub home {

	# build a simple home page
	$html  = $cgi->h1('Manage facets');
	$html .= $cgi->p('Use these pages to manage MyLibrary Portal facets.');

}


sub create {

	# initialize the output
	$html = $cgi->h1('Create');
	
	# branch according to the input
	if (! $cgi->param('button')) {
		
		# build the form
		$html .= $cgi->p('Use this form to create a facet.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'create');
		$html .= table({-border => 0}, Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item', 'Value']),
							td(['Facet', $cgi->textfield(-name => 'facet_name')]),
							td(['Note', $cgi->textarea(-name => 'facet_note', -rows => 7, -columns => 40)]),
							td([''    , $cgi->submit(-name => 'button', -value => 'Create')])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
    else {
    
    	# create an object, fill it, and save
    	my $facet = MyLibrary::Facet->new;
    	$facet->facet_name($cgi->param('facet_name'));
    	$facet->facet_note($cgi->param('facet_note'));
		$facet->commit;
				
		# echo the results
		$html .= $cgi->p('Facet created:');
		$html .= $cgi->ul(li('ID -', $facet->facet_id),
		                  li('Facet - ', $facet->facet_name),
		                  li('Note - ', $facet->facet_note)
		                 );
    
    }
    
}


sub find {

	# initialize the output
	$html  = $cgi->h1('All facets');
	$html .= $cgi->p('This is a list of all the facets:');
	
	# get all the facets, build a list, and display it
	my @facets = MyLibrary::Facet->get_facets (sort => 'name');
	my $items;
	foreach (@facets) {
	
		# create a list of terms associated with each facet
		my $terms = '<ol>';
		foreach my $id ($_->related_terms(sort => 'name')) {
		
			my $term = MyLibrary::Term->new(id => $id);
			$terms .= '<li>' . $cgi->a({-href => TERMCGI . '?cmd=edit&button=edit&name=' . $term->term_name}, $term->term_name) . ' - ' . $term->term_note . ' (' . $cgi->a({-href => RESOURCECGI . '?cmd=find&field=term&button=Go&query=' . $term->term_name}, scalar($term->related_resources) . ' items') . ')</li>';
		
		}
		$terms .= '</ol>';
		
		# build the list
		$items .= $cgi->li($cgi->a({-href=>$cgi->script_name . '?cmd=edit&button=edit&id=' . $_->facet_id}, $_->facet_name), ' - ', $_->facet_note, $terms);
		
	}
	$html .= $cgi->ol($items);
	
}


sub edit {

	# initialize the output and a button
	$html = $cgi->h1('Edit facets');
	my $submit .= $cgi->submit(-name => 'button', -value => 'Edit');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# create a hash of facets; needs to be sorted
		my @facets = MyLibrary::Facet->get_facets(sort => 'name');
		my %facets;
		foreach (@facets) { $facets{$_->facet_id} = $_->facet_name }
	
		# display a form
		$html .= $cgi->p('Select a facet to edit');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->popup_menu(-name => 'id', -values => \%facets);
		$html .= $submit;			 
		$html .= $cgi->endform;

	}
	
	elsif (! $cgi->param('facet_name') || ! $cgi->param('facet_note')) {
	
    	# create an object based on the input, and create form parts
    	my $facet = MyLibrary::Facet->new(id => $cgi->param('id'));
    	
		# display an edit form
		$html .= $cgi->p('Use this form to edit a facet.');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'edit');
		$html .= $cgi->hidden(-name => 'id', -value => $facet->facet_id);
		$html .= table({-border => 0},
					Tr({-align => 'left', -valign => 'top'},
						[
							th(['Item',  'Value']),
							td(['ID'  ,  $facet->facet_id]),
							td(['Facet', $cgi->textfield(-name => 'facet_name', -value => $facet->facet_name)]),
							td(['Note',  $cgi->textarea(-name => 'facet_note', -value => $facet->facet_note, -rows => 7, -columns => 40)]),
							td([''    ,  $submit])
						]
					)
				 );
		$html .= $cgi->endform;
    
    }
    
	else {

		# create and object, fill it, and save
		my $facet = MyLibrary::Facet->new(id => $cgi->param('id'));
		$facet->facet_name($cgi->param('facet_name'));
		$facet->facet_note($cgi->param('facet_note'));
		$facet->commit;
		
		# echo the result
		$html .= $cgi->p('Facet updated.');
		$html .= $cgi->p('ID:', $facet->facet_id, '<br />Facet:', $facet->facet_name, '<br />Note:' , $facet->facet_note);
    
    }

}


sub delete {

	# initialize the output
	$html = $cgi->h1('Delete facets');
	
	# branch according to the input
	if (! $cgi->param('button')) {
	
		# build of hash of all facets; needs to be sorted
		my @facets = MyLibrary::Facet->get_facets(sort => 'name');
		my %facets;
		foreach (@facets) { $facets{$_->facet_id} = $_->facet_name }

		# display a pop-up list of facets
		$html .= $cgi->p('Select a facet to delete:');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->popup_menu(-name => 'id', -values => \%facets);
		$html .= $cgi->submit(-name => 'button', -value => 'delete', -label => 'Delete');		 
		$html .= $cgi->endform;

	}
	
	elsif ($cgi->param('button') eq 'Delete') {
    
    	# create an object based on input
    	my $facet = MyLibrary::Facet->new(id => $cgi->param('id'));
    	
		# display a confirmation
		$html .= $cgi->p('Are you sure you want to delete the facet named <b>' . $facet->facet_name . '</b>?');
		$html .= $cgi->start_form(-method => 'post', -action => $cgi->script_name);
		$html .= $cgi->hidden(-name => 'cmd', -value => 'delete');
		$html .= $cgi->hidden(-name => 'id', -value => $facet->facet_id);
		$html .= $cgi->submit(-name => 'button', -value => 'yes', -label => 'Yes');
		$html .= ' ';
		$html .= $cgi->submit(-name => 'button', -value => 'no', -label => 'No');		 
		$html .= $cgi->endform;
    
    }
    
    elsif ($cgi->param('button') eq 'Yes') {
    
    	# delete the facet; do the work and echo the result
    	my $facet = MyLibrary::Facet->new(id => $cgi->param('id'));
    	$facet->delete;
    	$html .= $cgi->p('The facet ' . $facet->facet_name . ' has been deleted.');
    
    }
    
    elsif ($cgi->param('button') eq 'No') {
    
    	# abort
    	my $facet = MyLibrary::Facet->new(id => $cgi->param('id'));
    	$html .= $cgi->p('The facet ' . $facet->facet_name . ' has <b>not</b> been deleted.');

    }
    
}


sub gracefulExit {

	# output the 'magic line', a standard header, the content, and a standard footer, then done
	print $cgi->header;
	print &header;
	print $html;
	print &footer;
	exit;

}

