#!/usr/bin/perl

# people.cgi - a simple LDAP lookup script for the Google Box

# Eric Lease Morgan <emorgan@nd.edu>
# June 16, 2008 - added many constant definitions
# June 13, 2008 - added department
# May  19, 2008 - first cut

# require
use strict;
use CGI;
use CGI::Carp qw(fatalsToBrowser);
use Net::LDAP;

# define
use constant ATTRIBUTES => ( 'displayName', 'telephoneNumber', 'mail', 'ndprimaryaffiliation', 'title', 'uid', 'ndtoplevelprimarydepartment' );
use constant BASE       => 'o=University of Notre Dame,st=Indiana,c=US';
use constant DETAIL     => 'http://eds.nd.edu/cgi-bin/nd_ldap_search.pl?ldapfilter=uid=';
use constant MORE       => 'http://eds.nd.edu/cgi-bin/nd_ldap_search.pl?ldapfilter=cn=';
use constant SERVER     => 'directory.nd.edu';
use constant UPPER      => 5;

# initialize
my $cgi     = CGI->new;
my $query   = $cgi->param( 'query' );
my $index   = 0;
my $listing = '';
my $more    = '';

# search
my @results = &search( $query );

# re-initialize more
if (( $#results + 1 ) > UPPER ) { $more = MORE . "%2A$query%2A" }

# process each result
foreach ( @results ) {

	# initialize
	my ( $name, $telephone, $email, $affiliation, $title, $uid, $department ) = split "\t", $_;
	
	# build the detail
	my $detail = DETAIL . "$uid";
	
	# pretify the telephone number
	$telephone =~ s/\+1 (\d\d\d) /$1\//;
	$telephone =~ s/(\d\d\d) (\d\d\d\d)/$1-$2/;

	# build the result
	$listing .= '<MODULE_RESULT>';
	$listing .= "<Field name='name'>$name</Field>";
	$listing .= "<Field name='telephone'>$telephone</Field>";
	$listing .= "<Field name='email'>$email</Field>";
	$listing .= "<Field name='affiliation'>$affiliation</Field>";
	$listing .= "<Field name='title'>$title</Field>";
	$listing .= "<Field name='department'>$department</Field>";
	$listing .= "<Field name='detail'>$detail</Field>";
	if ( $#results > UPPER ) {
	
		$listing .= "<Field name='upper'>" . UPPER . "</Field>";
		$listing .= "<Field name='total'>" . ( $#results + 1 ) . "</Field>";
		$listing .= "<Field name='more'>$more</Field>";
		
	}
	$listing .= '</MODULE_RESULT>';
	
	# increment and check
	$index++;
	last if ( $index == UPPER );
	
}

# return results
print $cgi->header( -type=>'text/xml', -charset => 'UTF-8' );
print <<EOM;
<?xml version="1.0" encoding="UTF-8"?>
<OneBoxResults>$listing</OneBoxResults>
EOM

# quit
exit;


# do the hard work
sub search {

	# get the input
	my $query = $_[ 0 ];
	
	# connect and bind
	my $ldap = Net::LDAP->new( SERVER ) or die "$@";
	my $mesg = $ldap->bind;
	
	# search
	$mesg = $ldap->search( base => BASE, filter => "cn=*$query*", attrs => [ ATTRIBUTES ]);
	
	# sanity check
	$mesg->code && die $mesg->error;
	
	# echo; for debugging
	#foreach my $entry ($mesg->entries) { $entry->dump; }

	# process each hit
	my @results;
	foreach my $entry ($mesg->entries) {
	
		my $name        = $entry->get_value( 'displayName' )                 ? $entry->get_value( 'displayName' )                 : '';
		my $phone       = $entry->get_value( 'telephoneNumber' )             ? $entry->get_value( 'telephoneNumber' )             : '';
		my $email       = $entry->get_value( 'mail' )                        ? $entry->get_value( 'mail' )                        : '';
		my $affiliation = $entry->get_value( 'ndprimaryaffiliation' )        ? $entry->get_value( 'ndprimaryaffiliation' )        : '';
		my $title       = $entry->get_value( 'title' )                       ? $entry->get_value( 'title' )                       : '';
		my $uid         = $entry->get_value( 'uid' )                         ? $entry->get_value( 'uid' )                         : '';
		my $department  = $entry->get_value( 'ndtoplevelprimarydepartment' ) ? $entry->get_value( 'ndtoplevelprimarydepartment' ) : '';
		
		push @results, &escape_entities( "$name\t$phone\t$email\t$affiliation\t$title\t$uid\t$department" );
		
	}
	
	# cleanup and done
	$mesg = $ldap->unbind;
	return @results;

}


sub escape_entities {

	# get the input
	my $s = shift;
	
	# escape
	$s =~ s/&/&amp;/g;
	$s =~ s/</&lt;/g;
	$s =~ s/>/&gt;/g;
	$s =~ s/"/&quot;/g;
	$s =~ s/'/&apos;/g;

	# done
	return $s;
	
}

